%%
% To find the standing wave peak positions for big triangle from COMSOL 
% band structure for a given effective reflection and transmission 
% co-efficient at the corners
% 
% Inputs: tri.L: effective triangle side length, Gamma,
%         (All lengths are in number of unit cells)
%         Comsol band structures: experiment_compare_data_horizontal.mat, 
%         experiment_compare_data_slant.mat
%         Transmission and reflected co-efficients: (scat.t1, scat.r1) for horizontal
%         to slanted, (scat.t2, scat.r2) for slanted to slanted
%         shift_freq: real_peak_pos = scat_matrix_peak_pos + shift_freq
% 
% Outputs: Fig 1: function a in eigenvalue characteristic equation 
%          x^2+ax+1=0 (a=-2 is the peak position)
%          Fig 2: Lines at the peak position
%          Fig 3 and Fig 4: expected spectrum at horizontal and slanted
%          side respectively
%%

clear all;
close all;

% Load band structure data values
load('../Data/experiment_compare_data_horizontal.mat');
load('../Data/experiment_compare_data_slant.mat');

% Get the range of energy values for green region
energy_min = max(min(data_horz.energy), min(data_slant.energy));
energy_max = min(max(data_horz.energy), max(data_slant.energy));

% Get the range of energy values for gray region
energy_gray_min = min(data_slant.energy);
energy_gray_max = energy_min;

% Shift in spectrum
shift_freq = -3.523725502690544e6;

% Range of values of energy.green to check out for peak position
num_energy = 5e6;
energy.green = linspace(energy_min, energy_max, num_energy);
energy.gray = linspace(energy_gray_min, energy_gray_max, num_energy);

% Spectrum energy.green values
ene_spe_min = 318e6; % Range obtained from experiment data
ene_spe_max = 336.9e6;
ene_spe = linspace(ene_spe_min - shift_freq, ene_spe_max - shift_freq, ...
    1e3);

% Find the k and g0 value for each energy via interpolation
kinterp.green.horz = interp1(data_horz.energy, data_horz.k, ...
    energy.green);
kinterp.green.slant = interp1(data_slant.energy, data_slant.k, ...
    energy.green);

kinterp.gray.slant = interp1(data_slant.energy, data_slant.k, ...
    energy.gray);

g0interp.green.horz = interp1(data_horz.energy, data_horz.g0, ...
    energy.green);
g0interp.green.slant = interp1(data_slant.energy, ...
    data_slant.g0, energy.green);

g0interp.gray.slant = interp1(data_slant.energy, ...
    data_slant.g0, energy.gray, 'spline');

% Get g(beta) for each energy
tri.amech = 1.570496020158940e-05;
tri.L = 25.480436788; % side length
tri.realL = 28;
tri.amech_corr = pi/data_horz.k(end);
tri.Leff_corr = tri.L*tri.amech/tri.amech_corr;

% Reflection and transmission co-efficients
scat.t1 = sqrt(0.9);
scat.r1 = 1i*sqrt(1 - abs(scat.t1)^2);
scat.t2 = scat.t1;
scat.r2 = 1i*sqrt(1 - abs(scat.t2)^2);

% Get peak positions in green region
[peak_pos.green, kpeak.green, vel.green, g0peak.green] = ...
    get_peak_pos(kinterp.green, g0interp.green, tri, ...
    scat, energy.green);

% Get peak positions in gray region
[peak_pos.gray, kpeak.gray, vel.gray, g0peak.gray] = ...
    get_peak_pos_gray(energy_gray_min, energy_gray_max, ...
    tri, data_slant, scat);

%% Plotting spectrum

% Measurement node distance (calculated counter-clockwise from right corner)
num_points = ceil(tri.Leff_corr);
start = rem(tri.Leff_corr,1)/2;
finish = tri.Leff_corr - rem(tri.Leff_corr,1)/2;
dis_temp = linspace(start, finish, num_points)*tri.amech_corr;

dis = [dis_temp dis_temp+tri.Leff_corr*tri.amech_corr ...
    dis_temp+2*tri.Leff_corr*tri.amech_corr]; 

% Get amplitudes at a distance dis
for i=1:size(dis,2)
    % Green region
    amp.green(:,i) = get_amp_eigenvector(kpeak.green, ...
        scat, dis(i), tri, vel.green);
    
    % Gray region
    amp.gray(:,i) = get_amp_eigenvector_gray(kpeak.gray, ...
        scat, dis(i), tri, vel.gray);
end

amp.all = cat(1, amp.gray, amp.green(1:24,:));
peak_pos.all = horzcat(peak_pos.gray, peak_pos.green(1:24));

%% Plotting

figure()
hold on;
set(gca,'FontSize',18)
pbaspect([4.1538 1 1])
box on;

%% Line plot

% peak_index = 10;
% basevalue = 0;
% % maxvalue = max(max(amp(peak_index:peak_index+1, :)));
% 
% plot(28*dis/(tri.L*tri.amech), amp.all(peak_index, :).^2, ...
%         'LineWidth', 2, 'Color', 'blue')
% % plot(28*dis/(tri.L*tri.amech), amp.all(peak_index+1, :).^2, ...
%         'LineWidth', 2, 'Color', 'red')

%% Image plot

% xlabel('Distance from left bottom corner');
% ylabel('Peak index');
% xlim([0.5 75.5])
% ylim([0.5 size(peak_pos.all,2)+0.5])
% 
% % Image plot of all the modes
% image(amp.all.^2,'CDataMapping','scaled')
% 
% % Image plot of two modes
% % image(amp(peak_index:peak_index+1,:).^2,'CDataMapping','scaled')
% % colormap(flipud(gray))
% 
% colorbar;

%% Multiple eigenstates line plot (with subplots)

index_start = 10;
index_end = 18;
for peak_index = index_start:index_end
    s = subplot(index_end-index_start+1,1,peak_index-index_start+1);
    
    plot(amp.all(peak_index, :).^2, ...
        'LineWidth', 2, 'Color', 'blue', 'Marker', '.', ...
        'MarkerFaceColor', 'blue', 'MarkerSize', 10);

    if peak_index~=index_end
        set(s,'XTick',[])
    end
end
